'use client';
import { useQuery } from '@tanstack/react-query';
import { useMemo } from 'react';
import moment from 'moment';
import Icon from '@/components/Icon';
import { Button } from '@/components/UI/Button';
import { Checkbox } from '@/components/UI/Checkbox';
import { Input } from '@/components/UI/Input';
import { Select } from '@/components/UI/Select';
import Table, { type TableData, TableDataColumn } from '@/components/UI/Table';
import Operations, { type OperationKeys } from '@/components/Operations';
import { SimCardService } from '@/services/Settings/SimCardService';
import TableLoading from '@/components/UI/Loaders/TableLoading';
import { IOption } from '@/types/IOption';

const operations: OperationKeys = ['watch', 'copy', 'edit', 'remove'];

const headHeadings = [
  {
    name: <Checkbox isActive={false} value='all' />,
    key: 'ALL_CHECKBOXES',
    className: '[&_div]:justify-center',
    renderColumn: (_column: TableDataColumn, _rowData: TableData) => {
      return <Checkbox isActive={false} value={'1'} wrapperClassName='justify-center' />;
    },
  },
  {
    name: 'Операции',
    key: 'operations',
    renderColumn: (_column: TableDataColumn, _rowData: TableData) => {
      return <Operations operationKeys={operations} />;
    },
  },
  { name: 'Номер', key: 'number' },
  { name: 'Номер SIM карты', key: 'number_sim_card' },
  { name: 'Госномер', key: 'gos_number' },
  { name: 'Номер устройства', key: 'device_number' },
  { name: 'Статус', key: 'status' },
  { name: 'Время регистрации', key: 'registration_time' },
  { name: 'Компания', key: 'company' },
  { name: 'Город выпуска', key: 'city_of_issue' },
  { name: 'Сотовый оператор', key: 'mobile_operator' },
  { name: 'Дата окончания срока', key: 'expiration_date' },
  { name: 'Продавец карты', key: 'card_seller' },
  { name: 'MSISDN', key: 'msisdn' },
];

const statusChooseOptions: IOption[] = [
  { label: 'Все', value: 'ALL' },
  { label: 'Включено', value: 'enabled' },
  { label: 'Выключено', value: 'turned_off' },
];

const statusOptions: IOption[] = [
  { label: 'Все', value: 'ALL' },
  { label: 'Установлено', value: 'downloaded' },
  { label: 'Удалить', value: 'delete' },
];

const SimCard = () => {
  const { data, isPending } = useQuery({
    queryKey: ['getSimCardInfo'],
    queryFn: () => SimCardService.getSimCardInfo(),
  });

  const tableData = useMemo((): TableData[] | undefined => {
    if (data?.data.infos?.length) {
      const simCards = data.data.infos;
      return simCards.map((simCard, index) => {
        return {
          id: simCard.id,
          number: index + 1,
          number_sim_card: simCard.cardNum || '',
          gos_number: simCard.vehiIDNO || '',
          device_number: simCard.devIDNO || '',
          status: simCard.status === 1 ? 'Включено' : 'Не включено',
          registration_time: moment(simCard.registrationTime).format('YYYY-MM-DD HH:mm:ss'),
          company: simCard.company.name,
        };
      });
    }
  }, [data?.data.infos]);

  return (
    <div className='flex flex-col gap-6'>
      <div className='flex items-center gap-5'>
        <Select placeholder='Выбор статуса' options={statusChooseOptions} />
        <Select placeholder='Статус' options={statusOptions} />
        <Select placeholder='Выбор компании' />
        <Input
          placeholder='Номер SIM карты'
          wrapperInputClassName='h-[56px] min-w-[283px]'
          icon={
            <Icon
              src='/images/icons/icon-search.svg'
              className='size-[21px] min-w-[21px] text-primary mr-4'
            />
          }
        />
        <Button>Все SIM</Button>
        <Button background='green'>Добавить</Button>
        <Button background='red'>Удалить</Button>
        <Button variant='gray_outlined'>Импорт Excel</Button>
        <Button variant='gray_outlined'>Excel</Button>
      </div>

      {isPending ? <TableLoading /> : <Table headHeadings={headHeadings} data={tableData || []} />}
    </div>
  );
};

export default SimCard;
