'use client';
import { Fragment, useMemo } from 'react';
import { useQuery } from '@tanstack/react-query';
import Icon from '@/components/Icon';
import { Button } from '@/components/UI/Button';
import { Checkbox } from '@/components/UI/Checkbox';
import { Input } from '@/components/UI/Input';
import { Select } from '@/components/UI/Select';
import Table, { type TableData, TableDataColumn } from '@/components/UI/Table';
import Operations, { type OperationKeys } from '@/components/Operations';
import { DeviceService } from '@/services/Settings/DeviceService';
import TableLoading from '@/components/UI/Loaders/TableLoading';
import type { IOption } from '@/types/IOption';

const operations: OperationKeys = ['watch', 'copy', 'edit', 'remove'];

const headHeadings = [
  {
    name: <Checkbox isActive={false} value='all' />,
    key: 'ALL_CHECKBOXES',
    className: '[&_div]:justify-center',
    renderColumn: (_column: TableDataColumn, _rowData: TableData) => {
      return <Checkbox isActive={false} value={'1'} wrapperClassName='justify-center' />;
    },
  },
  {
    name: 'Операции',
    key: 'operations',
    renderColumn: (_column: TableDataColumn, _rowData: TableData) => {
      return <Operations operationKeys={operations} />;
    },
  },
  { name: 'Номер', key: 'number' },
  { name: 'Номер устройства', key: 'device_number' },
  { name: 'Номер SIM карты', key: 'number_sim_card' },
  { name: 'Серийный номер', key: 'serial_number' },
  { name: 'Идентификация устройства', key: 'device_id' },
  { name: 'Тип устройства', key: 'device_type' },
  { name: 'Марка', key: 'mark' },
  { name: 'Модель', key: 'model' },
  { name: 'Компания', key: 'company' },
  { name: 'Сопутствующее транспортное средство', key: 'related_vehicle' },
  { name: 'Дата привязки', key: 'binding_date' },
  { name: 'Принадлежащее транспортное средство', key: 'owned_vehicle' },
  { name: 'MAC Check Status ', key: 'mac_check_status' },
  { name: 'MAC Check Time', key: 'mac_check_time' },
];

const deviceData = [
  { title: 'Число подключений', key: 'connection_count' },
  { title: 'Незарегистрированных подключений', key: '' },
  { title: 'Управление GPS', key: 'gps_management' },
  { title: 'Количество добавленных', key: 'number_of_added' },
  { title: 'Управление Видео', key: 'video_management' },
  { title: 'NO. Of Used', key: 'no_of_used' },
];

const statusOptions: IOption[] = [
  { label: 'Все', value: 'ALL' },
  { label: 'Установлено', value: 'downloaded' },
  { label: 'Удалить', value: 'delete' },
];

const typeDeviceOptions: IOption[] = [
  { label: 'Все', value: 'ALL' },
  { label: 'Видеотерминал', value: 'video_terminal' },
  { label: 'Терминал местоположения', value: 'terminal_placement' },
  { label: 'Main defense terminal', value: 'main_defense_terminal' },
];

const typeOptions: IOption[] = [
  { label: 'Все', value: 'ALL' },
  { label: 'Госномер', value: 'gos_number' },
  { label: 'Device No', value: 'device_no' },
  { label: 'Производители', value: 'firms' },
  { label: 'Марка', value: 'mark' },
  { label: 'Модель', value: 'model' },
  { label: 'Серийный №', value: 'serial_number' },
  { label: 'Дата производства', value: 'production_date' },
  { label: 'Дата привязки', value: 'date_of_binding' },
  { label: 'MAC Addr', value: 'mac_addr' },
  { label: 'Версия ПО', value: 'po_version' },
  { label: 'Версия оборудования', value: 'device_version' },
  { label: 'Телефон', value: 'phone' },
  { label: 'Компания', value: 'company' },
  { label: 'Inspection Standard Specification', value: 'inspection_standard_specification' },
  { label: 'Inspection Qualified Date', value: 'inspection_qualified_date' },
  { label: 'Фактический номер SIM-карты', value: 'sim_card_number' },
];

const Device = () => {
  const { data, isPending, refetch, isFetching } = useQuery({
    queryKey: ['getDevices'],
    queryFn: () => DeviceService.getDevices(),
  });

  const tableData = useMemo((): TableData[] | undefined => {
    if (data?.data.infos?.length) {
      const devices = data.data.infos;
      return devices.map((device, index) => {
        return {
          id: device.id,
          number: index + 1,
          mark: device.brand || '',
          model: device.model || '',
          number_sim_card: device.simInfo || '',
          company: device.companyName || '',
          binding_date: device.stlTmStr,
          owned_vehicle: device.stlTmStr,
        };
      });
    }
  }, [data?.data.infos]);

  return (
    <div className='flex flex-col gap-6'>
      <div className='flex items-center gap-5'>
        <Select placeholder='Статус' options={statusOptions} />
        <Select placeholder='Тип устройства' options={typeDeviceOptions} />
        <Select placeholder='Выбор компании' />
        <Select placeholder='Выбор типа' options={typeOptions} />
        <Input
          placeholder='Введите чёткий запрос'
          wrapperInputClassName='h-[56px] min-w-[283px]'
          icon={
            <Icon
              src='/images/icons/icon-search.svg'
              className='size-[21px] min-w-[21px] text-primary mr-4'
            />
          }
        />
      </div>

      <div className='flex items-center gap-5'>
        <Button onClick={() => refetch()}>Все устройства</Button>
        <Button background='green'>Устройства в сети</Button>
        <Button background='darkYellow' className='min-w-max'>
          Незарегистрированные
        </Button>
        <Button background='green'>Добавить</Button>
        <Button background='red'>Удалить</Button>
        <Button variant='gray_outlined'>Настройка экспорта</Button>
        <Button variant='gray_outlined'>Настройка импорта</Button>
      </div>

      <div className='flex flex-wrap max-w-[60%]'>
        {deviceData.map((item) => (
          <Fragment key={item.key}>
            <p className='text-[16px] leading-[19px]'>
              {item.title}:<span className='font-bold text-primary'> 20</span>;&nbsp;
            </p>
          </Fragment>
        ))}
      </div>

      {isPending ? (
        <TableLoading />
      ) : (
        <Table isLoading={isFetching} headHeadings={headHeadings} data={tableData || []} />
      )}
    </div>
  );
};

export default Device;
