'use client';
import { FC, useMemo, useState } from 'react';
import clsx from 'clsx';
import Icon from '@/components/Icon';
import Tabs from '@/components/Tabs';
import Table from '@/components/UI/Table';
import { useFullWindowContext } from '@/contexts/FullWindowComponentContext';
import { IRecord } from '@/queries/records.interfaces';
import { groupBy } from '@/utils/object';
import { TimelineTable } from './TimelineTable';

const filterTabs: { title: string; key: string }[] = [
  {
    title: 'Время',
    key: 'time',
  },
  {
    title: 'Файл',
    key: 'file',
  },
];

const mockTimelineData = [
  {
    label: 'Все',
  },
  {
    label: 'Внутренняя',
  },
  {
    label: 'Левпер',
  },
  {
    label: 'Левзад',
  },
  {
    label: 'Правпер',
  },
  {
    label: 'Правзад',
  },
];

const theadHeadings = [
  { name: 'Операции', key: 'operation' },
  { name: 'Номер', key: 'number' },
  { name: 'Время', key: 'time' },
  { name: 'Тип тревоги', key: 'operation' },
  { name: 'Госномер', key: 'gos_number' },
  { name: 'Канал', key: 'channel' },
  { name: 'Расположение файла (MB)', key: 'file_path' },
  { name: 'Файл', key: 'file' },
  { name: 'Тэги', key: 'tags' },
  { name: 'Скачать файл', key: 'download' },
];

interface IProps {
  records: IRecord[];
  className?: string;
  onAddVideo: (rec: IRecord) => void;
}

export const Timeline: FC<IProps> = ({ className, records, onAddVideo }) => {
  const [activeTab, setActiveTab] = useState<keyof typeof tabsContent>('time');
  const { handleToggleComponent } = useFullWindowContext();

  const channels = useMemo(() => {
    return Object.values(groupBy(records, (item) => item.chn));
  }, [records]);

  const tabsContent = {
    time: <TimelineTable tracks={channels} onAddVideo={onAddVideo} />,
    file: <Table className='mx-4' headHeadings={theadHeadings} />,
  };

  return (
    <div
      className={clsx(
        'rounded-xl h-full bg-white border-[1px] border-light_gray pt-[14px]',
        className,
      )}
    >
      <div className='flex items-center justify-between pr-[10px] pl-[21px] pb-3'>
        <Tabs
          tabs={filterTabs}
          value={activeTab}
          onChange={(value) => setActiveTab(value as keyof typeof tabsContent)}
        />
        <div className='flex items-center gap-2'>
          <button
            type='button'
            className='w-10 h-10 flex justify-center items-center border-[1px] border-light_gray rounded-lg'
          >
            <Icon src='/images/icons/icon-server.svg' className='w-6 h-6 text-primary' />
          </button>
          <button
            type='button'
            className='w-10 h-10 flex justify-center items-center border-[1px] border-light_gray rounded-md'
          >
            <Icon src='/images/icons/icon-download.svg' className='w-6 h-6 text-primary' />
          </button>
          <button
            type='button'
            className='w-10 h-10 flex justify-center items-center border-[1px] border-light_gray rounded-md'
          >
            <Icon src='/images/icons/icon-widget.svg' className='w-6 h-6 text-primary' />
          </button>
          <button
            title='Открыть на весь экран'
            aria-label='Открыть на весь экран'
            type='button'
            className='w-10 h-10 flex justify-center items-center border-[1px] border-light_gray rounded-md'
            onClick={() => {
              handleToggleComponent(
                <Timeline records={records} className='max-w-[80%]' onAddVideo={onAddVideo} />,
              );
            }}
          >
            <Icon src='/images/full.svg' className='w-6 h-6 text-primary' />
          </button>
        </div>
      </div>
      {tabsContent[activeTab]}
    </div>
  );
};
