'use client';

import React, { useMemo, useState } from 'react';
import clsx from 'clsx';
import Tabs from '@/components/Tabs/Tabs';

import SquareButton from '@/components/SquareButton';
import Icon from '@/components/Icon';
import GridButton from '@/components/GridButton';
import { Table } from '@/components/Table/Table';

import { useVehicleGPSMultiple } from '@/queries/map';
import { useFullWindowContext } from '@/contexts/FullWindowComponentContext';
import type { ICompany } from '@/types/ICompnay';
import type { IVehicle } from '@/types/IVehicle';
import { useGetUserVehicleExForIndex } from '@/queries/records';
import { StandardVehicle } from '@/classes/StandardVehicle';

import { useAlarms } from '@/queries/tables';
import { groupBy } from '@/utils/object';
import { data, IRow, modal, tableHeadings } from './constants';

interface IProps {
  vehicles: StandardVehicle[];
  companies: ICompany[];
  className?: string;
}

const TablesCard: React.FC<IProps> = ({ vehicles, companies, className }) => {
  const [value, setValue] = useState<string>(data[0].key);
  const { handleToggleComponent } = useFullWindowContext();

  const onChange = (key: string) => {
    setValue(key);
  };

  const alarms = useAlarms();

  const rows = useMemo(() => {
    return (vehicles || []).map((el) => {
      return {
        company: companies?.find((cmp) => cmp.id === el.getParentID()),
        vehicle: el,
      } as IRow;
    });
  }, [companies, vehicles]);

  const alarmsByVehicles = useMemo(() => {
    const allAlarms = Object.values(alarms.data || {});

    return Object.values(groupBy(allAlarms, (el) => el.info.DevIDNO));
  }, [alarms.data]);

  const tableSource = useMemo(() => {
    const sources = {
      control: rows,
      alert: alarmsByVehicles,
      overspeed: alarmsByVehicles,
      fatigue: alarmsByVehicles,
      offset: alarmsByVehicles,
      zones: alarmsByVehicles,
      other: alarmsByVehicles,
      driving: alarmsByVehicles,
      media: alarmsByVehicles,
      system: alarmsByVehicles,
    };

    return (sources as any)[value];
  }, [value, rows, alarmsByVehicles]);

  return (
    <div
      className={clsx(
        className,
        'flex flex-col gap-3 bg-white h-full rounded-xl px-6 py-[11px] border-[1px]',
      )}
    >
      <div className='flex justify-between items-center'>
        <div className='max-w-[80%] overflow-x-auto scrollbar pb-2'>
          <Tabs tabs={data} value={value} onChange={onChange} />
        </div>
        <div className='flex gap-3'>
          <GridButton
            data={modal}
            classes={{
              card: 'right-0 top-[48px]',
            }}
          />
          <SquareButton
            classes={{
              root: 'h-[40px] w-[40px]',
            }}
            onClick={() =>
              handleToggleComponent(
                <TablesCard
                  vehicles={vehicles}
                  companies={companies}
                  className='max-w-full w-full'
                />,
              )
            }
          >
            <Icon src='/images/full.svg' className='h-[24px] w-[24px] text-primary' />
          </SquareButton>
        </div>
      </div>

      <div className='overflow-auto w-full scrollbar'>
        <Table
          columns={tableHeadings[value] || []}
          data={tableSource || []}
          className='-mx-5'
          headerClassName='bg-[#E3F6FE] text-[#2981B3] !min-h-[44px] border-none'
          itemClassName='bg-white border-none text-[16px]'
        />
      </div>
    </div>
  );
};

export default TablesCard;
