import classNames from 'classnames';
import moment from 'moment';
import { DashboardTodayResponseData } from '@/services/DashboardService';
import { Skeleton } from '@/components/UI/Loaders/Skeleton';
import { SingleAreaChartGraphic } from '@/pagesComponents/Dashboard/Graphics/SingleAreaChartGraphic';
import { PieChartGraphic } from '@/pagesComponents/Dashboard/Graphics/PieChartGraphic';
import { BarChartGraphic } from '@/pagesComponents/Dashboard/Graphics/BarChartGraphic';
import { info } from '@/pagesComponents/Dashboard/constants';
import { PaperWithGraphic } from '@/pagesComponents/Dashboard/PaperWithGraphic';
import { DashboardCart } from '@/pagesComponents/Dashboard/DashboardCart';
import { useDashboardDayData } from '@/queries/dashboard';
import { alarmSignalLabels, pieColors, vehiStatusLabels } from '@/constants/pie';
import './styles.css';

interface IProps {
  days: string;
}

export const TodayData: React.FC<IProps> = ({ days }) => {
  const { data: todayData, isLoading: todayLoading } = useDashboardDayData({
    searchDays: +days,
    vehiclesStatus: 0,
  });

  return (
    <div>
      <div className='mt-[30px] flex w-full justify-between items-center overflow-x-auto whitespace-nowrap scrollbar pb-2'>
        {todayLoading ? (
          new Array(info.day_data.info.length)
            .fill('')
            .map((_, index) => (
              <Skeleton
                key={index}
                className={`w-full ${info.day_data.classNames.wrapperClassname}`}
              />
            ))
        ) : (
          <>
            {todayData?.summary?.result === 0 &&
              todayData.summary.infos &&
              info.day_data.info?.map((inf) => {
                const infoData = (todayData?.summary as DashboardTodayResponseData).infos;
                const val = infoData[inf.key as keyof typeof infoData] || 0;
                return (
                  <DashboardCart
                    key={inf.key}
                    title={inf.title}
                    value={inf.format ? inf.format(val) : val}
                    className={classNames(
                      info.day_data.classNames.wrapperClassname,
                      'max-w-[218px]',
                    )}
                    contentClassName={`${info.day_data.classNames.contentClassname} ${inf.borderColor}`}
                  />
                );
              })}
          </>
        )}
      </div>

      <section className='mt-7 grid grid-cols-3 gap-8'>
        <PaperWithGraphic title='Онлайн-тренд транспортных средств'>
          <SingleAreaChartGraphic
            data={todayData?.basicPanel?.infos?.vehicleOnlineConditions || []}
            yKey='count'
            xKey='gpsDate'
            color='green'
            withXAxis
            withYAxis
            height={220}
            id='singleOnlineTrand'
            xFormat={(val) => moment(val).format('MMM DD')}
            labelFormatter={(item) => moment(item.gpsDate).format('MMM DD')}
            translations='Количество'
          />
        </PaperWithGraphic>
        <PaperWithGraphic title='Тревожный тренд'>
          <SingleAreaChartGraphic
            data={todayData?.basicPanel?.infos?.alarmThrends || []}
            yKey='totalAlarmCount'
            xKey='gpsDate'
            withXAxis
            withYAxis
            height={220}
            id='singleTrand'
            xFormat={(val) => moment(val).locale('ru').format('MMM DD')}
            labelFormatter={(item) => moment(item.gpsDate).locale('ru').format('MMM DD')}
            translations='Общее количество'
          />
        </PaperWithGraphic>
        <PaperWithGraphic title='Распределение сигналов тревоги'>
          <PieChartGraphic
            data={
              todayData?.basicPanel?.infos?.alarmCount.map((el, index) => ({
                value: el,
                title: alarmSignalLabels[index],
                colorPieCell: pieColors[index],
                name: 'test',
              })) || []
            }
            className='mt-5'
            height={239}
          />
        </PaperWithGraphic>
      </section>

      <section className='grid grid-cols-4 gap-[30px] justify-between mt-[30px]'>
        <PaperWithGraphic title='Эксплуатация автомобиля'>
          <PieChartGraphic
            data={
              todayData?.basicPanel?.infos?.vehiStatus.map((el, index) => ({
                value: el,
                title: vehiStatusLabels[index],
                colorPieCell: pieColors[index],
                name: 'test',
              })) || []
            }
            className='mt-5'
            height={239}
          />
        </PaperWithGraphic>
        <PaperWithGraphic title='Ранг тревог ТС'>
          <BarChartGraphic
            data={todayData?.alarmTop?.infos || []}
            yKeys={['totalAlarmCount']}
            xKey='vehiIdno'
            isMixBarChart
            height={210}
            labelFormatter={(item) => item.vehiIdno}
            className='barChart mt-5'
          />
        </PaperWithGraphic>
        <PaperWithGraphic title='Пробег(км)'>
          <BarChartGraphic
            data={todayData?.lichengTop?.lichengTop || []}
            yKeys={['licheng']}
            xKey='vehiIdno'
            isMixBarChart
            height={210}
            labelFormatter={(item) => item.vehiIdno}
            className='barChart mt-5'
          />
        </PaperWithGraphic>
        <PaperWithGraphic title='Рейтинг времени онлайн (час)'>
          <BarChartGraphic
            data={todayData?.workTop?.workTimeTop || []}
            yKeys={['workTime']}
            xKey='vehiIdno'
            isMixBarChart
            height={210}
            labelFormatter={(item) => item.vehiIdno}
            className='barChart mt-5'
          />
        </PaperWithGraphic>
      </section>
    </div>
  );
};
