import { FC, useMemo } from 'react';
import moment from 'moment';
import 'moment/locale/ru';

import {
  Area,
  AreaChart,
  CartesianGrid,
  ResponsiveContainer,
  Tooltip,
  XAxis,
  YAxis,
} from 'recharts';
import clsx from 'clsx';
import {
  NameType,
  Formatter,
  ValueType,
  Payload,
} from 'recharts/types/component/DefaultTooltipContent';
import { TotalGraphicProps } from '@/pagesComponents/Dashboard/Graphics/types';

interface SingleAreaChartGraphicProps<T extends Record<string, any>> extends TotalGraphicProps {
  data: T[];
  color?: 'green' | 'blue';
  withXAxis?: boolean;
  withYAxis?: boolean;
  id: string;

  xKey: keyof T;
  yKey: keyof T | Array<keyof T>;

  // tooltipPayload?: () => object;

  labelFormatter?: (payload: T) => React.ReactNode;
  xFormat?: (value: any, index: number) => string;

  formatter?: Formatter<any, any>;
  translations?: string;
}

const colors: Record<NonNullable<SingleAreaChartGraphicProps<any>['color']>, string> = {
  blue: '#2981B3',
  green: '#13DB9D',
};

moment.locale('ru');

export const SingleAreaChartGraphic = <T extends object>({
  data,
  color = 'blue',
  height = 107,
  withXAxis,
  withYAxis,
  xKey,
  yKey,

  xFormat,
  labelFormatter,

  className,
  formatter,
  id,
  translations,
}: SingleAreaChartGraphicProps<T>) => {
  const YArray = useMemo(() => {
    return Array.isArray(yKey) ? yKey : [yKey];
  }, [yKey]);
  const mainYKey = useMemo(() => {
    return YArray[0];
  }, [YArray]);

  return (
    <ResponsiveContainer
      width='100%'
      height={height}
      className={clsx(className, {
        'ml-[-30px]': withXAxis && withYAxis,
      })}
    >
      <AreaChart data={data}>
        {withXAxis && (
          <XAxis
            dataKey={xKey as string}
            dy={10}
            fontSize={9}
            color='#6B6B6B'
            tickFormatter={xFormat}
          />
        )}
        {withYAxis && <YAxis dataKey={mainYKey as string} fontSize={9} color='#6B6B6B' />}

        <defs>
          <linearGradient id={id} x1={0.4} x2={0.4} y1={0} y2={0.9}>
            <stop offset='5%' stopColor={colors[color]} stopOpacity={0.8} />
            <stop offset='100%' stopColor={colors[color]} stopOpacity={0} />
          </linearGradient>
        </defs>
        <CartesianGrid strokeDasharray='3 3' />

        <Tooltip
          labelFormatter={
            labelFormatter
              ? (_, item) => {
                  return !item[0] ? null : labelFormatter(item[0]?.payload);
                }
              : undefined
          }
          formatter={(value, name) => {
            const translatedName = translations || name;
            return [value, translatedName];
          }}
          includeHidden
        />

        {YArray.map((yKeyCurr, index) => (
          <Area
            key={index}
            type='monotone'
            dataKey={yKeyCurr as string}
            stroke={colors[color]}
            fill={`url(#${id})`}
            strokeWidth={5}
          />
        ))}
      </AreaChart>
    </ResponsiveContainer>
  );
};
