'use client';
import clsx from 'clsx';
import { forwardRef, LegacyRef, ReactNode } from 'react';
import Spinner from '@/components/UI/Loaders/Spinner';

export type TableDataColumn = string | number | Object | ReactNode;

export type TableData = Record<string, TableDataColumn>;

interface TableHeadHeadings {
  name: string | ReactNode;
  key: string;
  className?: string;
  renderColumn?: (column: TableDataColumn, rowData: TableData) => ReactNode;
}

export interface TableProps {
  headHeadings: TableHeadHeadings[];
  data?: TableData[];
  onRowClick?: (rowData: TableData) => void;
  className?: string;
  isLoading?: boolean;
}

const Table = forwardRef(function Table(
  { headHeadings, data, onRowClick, className, isLoading }: TableProps,
  ref,
) {
  return (
    <div
      className={clsx(' scrollbar max-w-full relative', className, {
        'overflow-x-hidden': isLoading,
        'overflow-x-auto': !isLoading,
      })}
    >
      {isLoading && (
        <div className='w-full h-full absolute top-0 left-0 bg-[#22222266] z-10 flex justify-center items-center rounded-t-lg'>
          <Spinner />
        </div>
      )}
      <table className='w-full' ref={ref as LegacyRef<HTMLTableElement>}>
        <thead>
          <tr className='bg-[#E3F6FE]'>
            {headHeadings.map((head) => (
              <th
                key={head.key}
                className={clsx(
                  'text-center py-[7px] text-[#2981B3] text-sm leading-[16px] font-normal w-[150px] min-w-[150px] first:rounded-tl-lg first:rounded-bl-lg last:rounded-tr-lg last:rounded-br-lg',
                  head?.className,
                )}
              >
                {head.name}
              </th>
            ))}
          </tr>
        </thead>
        <tbody>
          {!data || data.length === 0 ? (
            <tr className='relative h-20'>
              <td
                colSpan={headHeadings.length > 9 ? 9 : headHeadings.length}
                className='py-8 text-center text-gray-500 sticky left-0 top-0 w-full h-full'
              >
                Нет содержимого
              </td>
            </tr>
          ) : (
            data.map((tableRow) => (
              <tr
                key={tableRow.id?.toString()}
                className={clsx({ 'cursor-pointer': !!onRowClick })}
                onClick={() => onRowClick?.(tableRow)}
              >
                {headHeadings.map((head) => {
                  const column = tableRow[head.key];
                  return (
                    <td
                      key={head.key}
                      className={clsx(
                        'text-center text-[#18120F] py-4 text-sm leading-[16px] font-normal w-[150px] min-w-[150px]',
                        tableRow?.className,
                      )}
                    >
                      {head?.renderColumn
                        ? head.renderColumn(column, tableRow)
                        : (column as string | number | ReactNode)}
                    </td>
                  );
                })}
              </tr>
            ))
          )}
        </tbody>
      </table>
    </div>
  );
});

export default Table;
