'use client';
import clsx from 'clsx';
import { FC, ReactNode, useCallback, useEffect } from 'react';
import { createPortal } from 'react-dom';
import { useScrollLock } from '@/utils/useScrollLock';
import Icon from '@/components/Icon';

export interface ModalProps {
  isOpen: boolean;
  closeModal: () => void;
  children: ReactNode;
  className?: string;
  bodyClassName?: string;
  title?: string;
  withCloseButton?: boolean;
  headerClassName?: string;
}

const Modal: FC<ModalProps> = ({
  children,
  closeModal,
  isOpen,
  className,
  bodyClassName,
  headerClassName,
  title,
  withCloseButton = true,
}) => {
  const { lockScroll, unlockScroll } = useScrollLock();

  const escFunction = useCallback(
    (event: KeyboardEvent) => {
      if (event.key === 'Esc' || event.key === 'Escape') {
        closeModal();
      }
    },
    [closeModal],
  );

  useEffect(() => {
    if (isOpen) {
      lockScroll();
    } else {
      unlockScroll();
    }
  }, [isOpen, unlockScroll, lockScroll]);

  useEffect(() => {
    document.addEventListener('keydown', escFunction, false);

    return () => {
      document.removeEventListener('keydown', escFunction, false);
    };
  }, [escFunction]);

  return createPortal(
    <>
      {isOpen && (
        <div
          className={clsx(
            'fixed top-0 left-0 z-[999] w-full h-full overflow-x-hidden overflow-y-auto bg-[#2223] backdrop-blur-sm',
            {
              modal_animation: isOpen,
            },
          )}
          aria-hidden='true'
          onClick={closeModal}
        >
          <div className='flex items-center justify-center w-full h-full'>
            <div
              className={clsx(
                'max-w-[1000px] w-full rounded-xl border border-light_gray shadow-md bg-white max-h-[80vh] overflow-y-auto scrollbar',
                className,
              )}
              onClick={(e) => {
                e.stopPropagation();
              }}
            >
              {(title || withCloseButton === true) && (
                <div
                  className={clsx(
                    'flex items-center gap-2 px-8 py-6 rounded-t-xl bg-[#15223C]',
                    {
                      'justify-end': !title,
                      'justify-between': !!title,
                    },
                    headerClassName,
                  )}
                >
                  {title && <h3 className='text-white font-medium text-2xl leading-6'>{title}</h3>}
                  {withCloseButton === true && (
                    <button
                      type='button'
                      onClick={closeModal}
                      className='size-10 min-w-10 min-h-10 rounded-[50%] flex justify-center items-center hover:bg-primary transition-colors'
                    >
                      <Icon src='/images/close.svg' className='size-6 text-white' />
                    </button>
                  )}
                </div>
              )}

              <div className={clsx('px-8 py-4', bodyClassName)}>{children}</div>
            </div>
          </div>
        </div>
      )}
    </>,
    document.body,
  );
};

export default Modal;
