import clsx from 'clsx';
import { ButtonHTMLAttributes, DetailedHTMLProps, FC } from 'react';
import Spinner from '@/components/UI/Loaders/Spinner';

interface IButtonProps
  extends DetailedHTMLProps<ButtonHTMLAttributes<HTMLButtonElement>, HTMLButtonElement> {
  background?: 'white' | 'primary' | 'blue' | 'green' | 'orange' | 'red' | 'black' | 'darkYellow';
  variant?: 'primary' | 'secondary' | 'gray_outlined' | 'green_outlined';
  isLoading?: boolean;
}

const backgroundVariants: Record<NonNullable<IButtonProps['background']>, string> = {
  primary: 'bg-primary',
  white: 'bg-white',
  blue: 'bg-[#46B5F8]',
  green: 'bg-[#13DB9D]',
  orange: 'bg-[#FF8548]',
  red: 'bg-[#E74A51]',
  black: 'bg-[#000000]',
  darkYellow: 'bg-[#DB8E13]',
};

const variants: Record<NonNullable<IButtonProps['variant']>, string> = {
  primary: 'text-base text-white',
  secondary: 'bg-transparent border-text text-text border-[1px]',
  gray_outlined: 'bg-white border border-light-gray text-[#AFAFAF]',
  green_outlined: 'bg-white border border-[#13DB9D] text-[#13DB9D]',
};

export const Button: FC<IButtonProps> = ({
  type = 'button',
  className,
  children,
  background = 'primary',
  disabled,
  variant = 'primary',
  isLoading,
  ...rest
}) => {
  return (
    <button
      className={clsx(
        backgroundVariants[background],
        variants[variant],
        disabled && 'bg-opacity-65',
        'h-[55px] px-4 py-6 w-full flex items-center justify-center rounded-xl min-w-[55px] transition-transform duration-300 font-medium',
        className,
      )}
      type={type}
      disabled={disabled}
      {...rest}
    >
      {isLoading ? <Spinner /> : children}
    </button>
  );
};
