'use client';

import React, { useCallback, useRef, useState } from 'react';
import clsx from 'clsx';

import SquareButton from '@/components/SquareButton';
import Icon from '@/components/Icon';
import { Select } from '@/components/UI/Select';

import { useForm } from '@/utils/useForm';
import { useClickOutside } from '@/hooks/useClickOutside';
import { IBooleanField, ISelectField } from './constants';

const validation = {};

interface IClasses {
  card?: string;
  label?: string;
  button?: string;
  wrapperButton?: string;
}

interface IProps {
  data: Array<ISelectField | IBooleanField>;
  classes?: IClasses;
}

const GridButton: React.FC<IProps> = ({ data = [], classes = {} }) => {
  const wrapperRef = useRef<HTMLDivElement>(null);
  const handleSubmit = useCallback(async () => {}, []);

  const form = useForm(
    handleSubmit,
    validation,
    Object.fromEntries(data.map((item) => [item.key, item.defaultValue])) as Record<string, string>,
  );

  const [isOpen, setOpen] = useState(false);
  const handleClick = () => {
    if (data.length) {
      setOpen(!isOpen);
    }
  };

  useClickOutside(wrapperRef, () => setOpen(false));

  return (
    <div className='cursor-pointer relative'>
      <SquareButton
        classes={{
          root: clsx('h-[40px] w-[40px]', classes.wrapperButton, {
            'bg-primary border-primary': isOpen,
          }),
        }}
        onClick={handleClick}
      >
        <Icon
          src={isOpen ? '/images/close.svg' : '/images/grid.svg'}
          className={clsx('h-[24px] w-[24px] text-primary', {
            'text-white': isOpen,
          })}
        />
      </SquareButton>

      {isOpen && (
        <div
          className={clsx(
            'w-fit h-fit flex flex-col gap-3 p-4 bg-white border-light_gray border-[1px] absolute right-[-200%] rounded-xl transition-all fadeIn',
            classes.card,
          )}
          ref={wrapperRef}
        >
          {data.map((item, index) => (
            <div key={index} className='flex justify-between items-center'>
              <div
                className={clsx(
                  'min-w-[240px] text-[14px] leading-[14px] max-w-[240px]',
                  classes.label,
                )}
              >
                {item.label}
              </div>

              {item.type == 'boolean' && (
                <div className='flex'>
                  {item.property.map((el, count) => (
                    <SquareButton
                      key={count}
                      onClick={() => {
                        form.handleChangeField(item.key)(el.value);
                      }}
                      classes={{
                        root: clsx(
                          'h-[40px] w-[85px] text-[14px] text-text hover:bg-secondary hover:text-primary border-none',
                          classes.button,
                          {
                            'bg-secondary': el.value === form.formData[item.key],
                          },
                        ),
                      }}
                    >
                      {el.label}
                    </SquareButton>
                  ))}
                </div>
              )}

              {item.type == 'select' && (
                <Select
                  options={item.property}
                  classes={{
                    input: '!h-[40px] min-w-[170px]',
                  }}
                  value={form.formData[item.key]}
                  onChange={form.handleChangeField(item.key)}
                />
              )}
            </div>
          ))}
        </div>
      )}
    </div>
  );
};

export default GridButton;
